/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: stream.h,v 1.33.2.3 2003/07/18 00:44:45 bazsi Exp $
 *
 ***************************************************************************/

#ifndef ZORP_STREAM_H_INCLUDED
#define ZORP_STREAM_H_INCLUDED

#include <zorp/zorplib.h>

#include <time.h>
#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifdef G_OS_WIN32
  #define SHUT_RDWR SD_BOTH
  #define SHUT_WR SD_SEND
  #define SHUT_RD SD_RECEIVE
#endif

#define Z_STREAM_MAX_NAME   64

#define Z_STREAM_FLAG_READ  1
#define Z_STREAM_FLAG_WRITE 2
#define Z_STREAM_FLAG_PRI   4
#define Z_STREAM_FLAG_TIMEOUT 8

#define Z_STREAM_TIMEOUT_BLOCK 0

/* Stream types */

#define ZST_FD     (0x0100)
#define ZST_PACKET (0x0200)
#define ZST_LINE   (0x0300)
#define ZST_SSL    (0x0400)
#define ZST_BUF    (0x0500)

/* stream control messages:

     0. byte message
     1. byte stream type
     3. byte undefined
     4. byte stream flags
 */
 
#define ZST_CTRL_MSG(x)       ((x)&0xFFFF)
#define ZST_CTRL_MSG_FLAGS(x) ((x) & 0xFF000000)

#define ZST_CTRL_MSG_FORWARD  0x80000000

#define ZST_CTRL_GET_FD               (0x01)
#define ZST_CTRL_SET_COND_READ        (0x02)
#define ZST_CTRL_SET_COND_WRITE       (0x03)
#define ZST_CTRL_SET_COND_PRI         (0x04)
#define ZST_CTRL_SET_CALLBACK_READ    (0x06)
#define ZST_CTRL_SET_CALLBACK_WRITE   (0x07)
#define ZST_CTRL_SET_CALLBACK_PRI     (0x08)
#define ZST_CTRL_SET_TIMEOUT_BLOCK    (0x0A)
#define ZST_CTRL_GET_COND_READ        (0x0C)
#define ZST_CTRL_GET_COND_WRITE       (0x0D)
#define ZST_CTRL_GET_COND_PRI         (0x0E)
#define ZST_CTRL_GET_CALLBACK_READ    (0x10)
#define ZST_CTRL_GET_CALLBACK_WRITE   (0x11)
#define ZST_CTRL_GET_CALLBACK_PRI     (0x12)
#define ZST_CTRL_SET_NONBLOCK         (0x14)

struct _ZStream;

typedef gboolean (*ZStreamCallback)(struct _ZStream *, GIOCondition, gpointer);

typedef struct _ZStreamSetCb
{
  ZStreamCallback cb;
  gpointer cb_data;
  GDestroyNotify cb_notify;
} ZStreamSetCb;

typedef struct _ZStreamFuncs
{
  GIOStatus (*read) (struct _ZStream  *stream,
                               gchar  *buf,
                               gsize   count,
                               gsize  *bytes_read,
                              GError **err);

  GIOStatus (*write) (struct _ZStream  *stream,
                          const gchar  *buf,
                                gsize   count,
                                gsize  *bytes_written,
                               GError **err);

  GIOStatus (*read_pri) (struct _ZStream  *stream,
                                   gchar  *buf,
                                   gsize   count,
                                   gsize  *bytes_read,
                                  GError **err);

  GIOStatus (*write_pri) (struct _ZStream  *stream,
                              const gchar  *buf,
                                    gsize   count,
                                    gsize  *bytes_written,
                                   GError **err);

  GIOStatus  (*shutdown) (struct _ZStream  *stream,
                                      int   i,
                                   GError **err);

  GIOStatus  (*close) (struct _ZStream  *stream,
                                GError **err);
  
  gboolean (*ctrl) (struct _ZStream *stream,
                              guint  function,
                           gpointer  value,
                              guint  vlen);

  void (*attach_source) (struct _ZStream *stream, GMainContext *context);
  void (*detach_source) (struct _ZStream *stream);

  gboolean (*watch_prepare)  (struct _ZStream *s, GSource *src, gint *timeout);
  gboolean (*watch_check)    (struct _ZStream *s, GSource *src);
  gboolean (*watch_dispatch) (struct _ZStream *s, GSource *src);
  void (*watch_finalize) (struct _ZStream *s, GSource *source);

  void (*freefn) (struct _ZStream *stream);

} ZStreamFuncs;

/*+

  ZStream encapsulates a Unix file descriptor. It contains some
  attributes and callbacks which are used by ZPoll.

  +*/
typedef struct _ZStream 
{
  ZStreamFuncs *funcs;
  gchar name[Z_STREAM_MAX_NAME];
  gint type;
  gint timeout;
  gint poll_timeout;

  time_t time_open;
  guint64 bytes_recvd, bytes_sent;
  
/* private, dont use them directly */

  guint ref;
  gboolean want_read;       /*+ do we want read callbacks? +*/
  gpointer user_data_read;  /*+ opaque pointer, can be used by read callback +*/
  GDestroyNotify user_data_read_notify;
  ZStreamCallback read_cb;  /*+ pointer to read callback +*/

  gboolean want_pri;        /*+ do we want urgent data callbacks? +*/
  gpointer user_data_pri;   /*+ opaque pointer, can be used by pri callback +*/
  GDestroyNotify user_data_pri_notify;
  ZStreamCallback pri_cb;   /*+ pointer to urgent data callback +*/

  gboolean want_write;      /*+ do we want write callbacks +*/
  gpointer user_data_write; /*+ opaque pointer, can be used by write callback +*/
  GDestroyNotify user_data_write_notify;
  ZStreamCallback write_cb; /*+ pointer to write callback +*/

  struct _ZStream *parent;

  GSource *source;
  GStaticMutex lock;

} ZStream;

typedef struct _ZStreamSource
{
  GSource super;
  ZStream *stream;
} ZStreamSource;

void z_stream_init(ZStream *s, gint type, ZStreamFuncs *funcs, gchar *name);
void z_stream_ref(ZStream *stream);
void z_stream_unref(ZStream *stream);

gboolean z_stream_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen);

void z_data_dump(char *prefix, const char *buf, guint len);

GSource *z_stream_source_new(ZStream *stream);

static inline gint 
z_stream_get_type(ZStream *s)
{
  return s->type;
}

#define z_stream_ctrl(s, f, v, l) (s)->funcs->ctrl(s, f, v, l)

static inline int
z_stream_get_fd(ZStream *s)
{
  gint ret = -1;
  if (!z_stream_ctrl(s, ZST_CTRL_GET_FD, &ret, sizeof(ret)))
    ret = -1;
  return ret;
}

static inline gboolean
z_stream_set_cond(ZStream *s, guint type, gboolean value)
{
  gboolean ret = FALSE;

  switch(type)
    {
    case Z_STREAM_FLAG_READ:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_COND_READ, &value, sizeof(value));
      break;
    case Z_STREAM_FLAG_WRITE:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_COND_WRITE, &value, sizeof(value));
      break;
    case Z_STREAM_FLAG_PRI:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_COND_PRI, &value, sizeof(value));
      break;
    default:
      break;
    }
  return ret;
}

static inline gboolean
z_stream_set_callback(ZStream *s, guint type, ZStreamCallback callback, gpointer user_data, GDestroyNotify notify)
{
  gboolean ret = FALSE;
  ZStreamSetCb cb;

  cb.cb = callback;
  cb.cb_data = user_data;
  cb.cb_notify = notify;

  switch(type)
    {
    case Z_STREAM_FLAG_READ:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_CALLBACK_READ, &cb, sizeof(cb));
      break;
    case Z_STREAM_FLAG_WRITE:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_CALLBACK_WRITE, &cb, sizeof(cb));
      break;
    case Z_STREAM_FLAG_PRI:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_CALLBACK_PRI, &cb, sizeof(cb));
      break;
    default:
      break;
    }
  return ret;
}

static inline gboolean
z_stream_set_timeout(ZStream *s, guint dummy G_GNUC_UNUSED /* for binary compat */, gint value)
{
  s->timeout = value;
  return TRUE;
}

static inline gboolean
z_stream_set_nonblock(ZStream *s, gboolean nonblock)
{
  return s->funcs->ctrl(s, ZST_CTRL_SET_NONBLOCK, &nonblock, sizeof(nonblock));
}

/* FIXME: make these inline functions */

#define z_stream_read(s, b, l, r, e)  (s)->funcs->read(s, b, l, r, e)
#define z_stream_write(s, b, l, r, e) (s)->funcs->write(s, b, l, r, e)

#define z_stream_read_pri(s, b, l, r, e)  (s)->funcs->read_pri(s, b, l, r, e)
#define z_stream_write_pri(s, b, l, r, e) (s)->funcs->write_pri(s, b, l, r, e)

#define z_stream_shutdown(s, i, e) (s)->funcs->shutdown(s, i, e)
#define z_stream_close(s, e)   (s)->funcs->close(s, e)

#define z_stream_attach_source(s, c)  (s)->funcs->attach_source(s, c)
#define z_stream_detach_source(s)     (s)->funcs->detach_source(s)

#ifdef __cplusplus
}
#endif

#endif
